// references
// https://www.youtube.com/watch?v=FUWLYC1z1LU

/*
To assist in development messages are logged in the developers console
*/


const url = 'webauthn.php';
const website_name = "Webauthn demo";

const show = 1;
const hide = 0;

async function start_webAuthn()
{
  centerElement(); // centre all the webauthn elements
  window.addEventListener('resize', centerElement);

  document.getElementById('wb_txt_browser').firstElementChild.textContent =  getBrowserInfo();

  if ((localStorage.webauthn_AAGUID === null) || (localStorage.webauthn_AAGUID == undefined))
	  localStorage.webauthn_AAGUID = 'unknown';
	  
  sessionStorage.webAuthn_authorised	= 'not yet';
  
  ShowObjectWithEffect('layer_register_credential', hide, 'fade', 1); 
  ShowObjectWithEffect('layer_enter_passkey', hide, 'fade', 1); 
  ShowObjectWithEffect('bnWebAuthn', hide, 'cliphorizontal', 300); 
  
  ShowObjectWithEffect('layer_login', show, 'fade', 1); 
  
  if (!window.PublicKeyCredential) { /* Browser not capable. Handle error. */ 
    ShowObjectWithEffect('layer_noWebAuthn', show, 'slidedown', 700); 
	return;
  }

  if (localStorage.webauthn_username != undefined)
    document.getElementById('username').value = localStorage.webauthn_username;
  else 
    localStorage.webauthn_username = '';
  
  if (localStorage.webauthn_userid != undefined)
    document.getElementById('userid').value = localStorage.webauthn_userid;
 
  document.getElementById('username').focus();

}
 
function end_webAuthn()
{
 document.getElementById('cbRegister').checked = false;

 ShowObjectWithEffect('layer_login', hide, 'cliphorizontal', 700); 
 ShowObjectWithEffect('bnWebAuthn', show, 'cliphorizontal', 300); 
}

function webAuthn_login_change(localElement,elementValue)
{
  localStorage.setItem(localElement, elementValue);
}

function webAuthn_login_userid(element)
{
  element.value = element.value.toLowerCase()
}

function webAuthn_login_Enter(event)
{ if (event.key === "Enter")
  { event.preventDefault();	  
	webAuthn_user_login();
  }	
}

async function webAuthn_user_login()
{ const functionName = "user_login: ";

 const validateEmail = (email) => {
  return String(email)
    .match(      /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|.(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/
    );
 };

 if (localStorage.webauthn_userid == undefined){
	  alert('Enter email address ....')
      return;
  }	  

  userid = localStorage.webauthn_userid;  
  
  if (!validateEmail(userid))
  { alert('Invalid email address ....');
    document.getElementById('userid').focus();
    return;
  }
  
  data = await getChallenge(userid);  // Credential Id and Challenge
  if (!data) // something is wrong - login aborted
	  return;
 
  challenge = data['challenge'];
  credentialId = data['credentialid']; // link to private key

  logMessage(functionName + 'credentialId: ' + credentialId);

  if (challenge.substring(0,5) == "error")  // database problem
  { alert(challenge);
    return;
  }	  

  const hostname = window.location.hostname;
  const rpID = hostname.startsWith("www.") ? hostname.slice(4) : hostname;
 
  if ((credentialId == "CID missing") || // Credential ID does not exist
        (document.getElementById('cbRegister').checked))
    { await registerCredential(rpID, userid, challenge);  }
  else
    { await passkey_login(rpID, userid, challenge, credentialId);  }
  
  if (sessionStorage.webAuthn_authorised == 'yes')
  { end_webAuthn();
    main(); // start website
  }

}

async function registerCredential(rpID, userID, challenge) {
  const functionName = "registerCredential: ";
  ShowObjectWithEffect('layer_register_credential', show, 'puff', 200); 
  
  userName = localStorage.webauthn_username;
  challengeArray = base64ToUint8Array(challenge);
  
  const options = {
// The challenge is produced by the server; 
    challenge: challengeArray,

  // Relying Party:
  rp: {
     name: website_name,
     id: rpID
    },

  // User:
  user: {
//    id: new TextEncoder().encode(userID), or Uint8Array.from(userID, c=>c.charCodeAt(0)),
    id: Uint8Array.from(userID, c=>c.charCodeAt(0)),
    name: userID,
    displayName: userName, 	
  },

/*	
Credential ID: Unique identifier for the credential	
Key Handle: Binary representation of the private key

see www,iana.org	
Algorithm Value - Name				- Hash
	-7			ECDSA				SHA-256
	-257		RSASSA-PKCS1-v1_5	SHA-256
	-258		RSASSA-PKCS1-v1_5	SHA-384
	-259		RSASSA-PKCS1-v1_5	SHA-512
	-37			RSASSA-PSS			SHA-256
	-38			RSASSA-PSS			SHA-384
	-39			RSASSA-PSS			SHA-512
	-47			EdDSA				n/a
*/

  pubKeyCredParams: [
    { type: "public-key", alg: -7}, // ES256 (error: if removed)
    { type: "public-key", alg: -257} // RSASSA-PKCS1-v1_5 with SHA-256 (RS256) - required for windows hello
	],

  authenticatorSelection: {
//    authenticatorAttachment: "platform", // probably better to not specify
//  residentKey: "discouraged", // optional (default), discouraged, required, preferred
    userVerification: "preferred" // preferred, discouraged, required
  },

  timeout: 120000,  // 2 minutes
  excludeCredentials: [], // nothing to exclude on 1st time registration
  attestation: "direct" // None, indirect, direct, enterprise
  };
	
// Note: The following call will cause the authenticator to display UI.
   try {
    credentialInfo = await navigator.credentials.create({ publicKey: options });

	const response = credentialInfo.response;
 
// credentialInfo.id - it's not guaranteed to be base64url-encoded
// Convert credentialId to base64url
    const rawId = new Uint8Array(credentialInfo.rawId);	
	const publicKeyData = response.getPublicKey();

    const attestationBuffer = response.attestationObject;
    const aaguid = extractAAGUID(attestationBuffer);
    const clientDataJSON = response.clientDataJSON; 

	await sendPublicKeyToServer( userID, challenge, rawId, clientDataJSON, publicKeyData, aaguid);
	 	
  } catch(err){
 // No acceptable authenticator or user refused consent. Handle appropriately.
    await removeChallenge(userID, challenge);  
   	
    showPasskeyError(err); 
    ShowObjectWithEffect('layer_register_credential', hide, 'fade', 1); 
  };
}

async function getChallenge(userid) { 
  const functionName = "getChallenge: ";

  logMessage(functionName +'start');
 
  const params = new URLSearchParams([
    ['option', 'getChallenge'],['userId', userid],['aaguid', localStorage.webauthn_AAGUID] ]);

  try {
    const response = await fetch(url + "/?" + params.toString());

    if (!response.ok) {
      const errorText = await response.text();
      logError(functionName +'Error 1:' + response.status + response.message);
      logError('Response body:' + errorText);
      throw new Error(functionName +'Network response was not ok');
    }

    const data = await response.json(); 
	
	logMessage(functionName +'done ok');
	return data;
	
  } catch (error) {
    if (error instanceof TypeError && error.message === 'Failed to fetch') {
       logError('Network or CORS error: ' + error);
	   alert("Network slow: try again");
    } else {
       logError(functionName +'Error 2: ' + error);
       alert(error);
    }

	return false;
  }
  
}

async function removeChallenge(userid, challenge) 
{ const functionName = "removeChallenge: ";
  logMessage(functionName +'start');
  
  const data = {
		option: 'removechallenge',
        userId: userid,
		challenge: challenge       
  };

 // Send the data to the server using Fetch API
  return fetch(url, {
      method: 'POST',
      headers: {
          'Content-Type': 'application/json'
      },
      body: JSON.stringify(data)
  })
  .then(response => {
      if (!response.ok) {
          throw new Error(functionName + 'Error 1! Status: ${response.status}'); }
	  
        return response.json();
    })
  .then(data => {
      if (data["status"] == "error")
		{ alert(functionName + 'Error: 2: ' + data["message"]);}
	  else
		{ logMessage(functionName + 'status: ' + data.status);			
          logMessage(functionName + 'message: ' + data.message);			
		}  
    })
  .catch(error => {
        logMessage(functionName + 'Error 3: ' + error);
    });
}

async function passkey_login(rpID, userid, challenge, credentialIdBase64url)
{
  functionName = 'passkey_login: ';	
  ShowObjectWithEffect('layer_enter_passkey', show, 'fold', 200); 
  const credentialId = Uint8Array.from(atob(credentialIdBase64url.replace(/-/g, '+').replace(/_/g, '/')),c => c.charCodeAt(0));

  logMessage(functionName + 'start');

  const publicKey = {
    challenge: base64ToUint8Array(challenge),

    rp: {
     name: website_name,
     id: rpID
    },
    allowCredentials: [{
      type: 'public-key',
	  id: credentialId
      // transports: ['internal'] // or ['usb', 'nfc', 'ble'] depending on authenticator
    }],
    timeout: 60000,
	// mediation: silent, // silent, optional, conditional - not always acceptable
    userVerification: 'preferred' // discouraged, preferred, required, optional
  };

  logMessage(functionName + 'get signature+');

  try {
    const assertion = await navigator.credentials.get({publicKey }); // mediation: 'silent' - cant get to work
    logMessage(functionName + 'Assertion:' + assertion);
	
// Extract the assertion signature
    const response = assertion.response;
    const signature = response.signature; // This is the assertion signature
    const authenticatorData = response.authenticatorData; // Additional data
    const clientDataJSON = response.clientDataJSON; // Client data

    logMessage(functionName + "Signature:" + signature);
    logMessage(functionName + "ClientDataJSON:" + clientDataJSON);
    logMessage(functionName + "authenticatorData:" + authenticatorData);

// Send the signature and other data to the server for verification
	await sendSignatureToServer( userid, challenge, signature, clientDataJSON, authenticatorData);
	
  } catch (err) {
	 showPasskeyError(err); 
     ShowObjectWithEffect('layer_enter_passkey', hide, 'fade', 1); 
  }
}


function sendPublicKeyToServer(userHandle, challenge, rawId, clientDataJSON, publicKeyData, aaguid ) {
    const functionName = 'sendPublicKeyToServer: ';
    logMessage(functionName + 'Start');			
	
    // Convert the public key to a format suitable for sending to the server
    const base64publicKey = btoa(String.fromCharCode(...new Uint8Array(publicKeyData)));
    const base64credentialId = btoa(String.fromCharCode(...rawId)).replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
	const base64clientDataJSON = base64urlEncode(clientDataJSON);
	
    if (localStorage.webauthn_AAGUID == "unknown")
	  localStorage.webauthn_AAGUID = aaguid; // used to denote passkey has been registered
    else // check if the aaguid has changed
	{ if (localStorage.webauthn_AAGUID != aaguid)
	  {	 logMessage(functionName +"localStorage AAGUID: " + localStorage.webauthn_AAGUID);
		  aaguid = localStorage.webauthn_AAGUID;
	  }	  
	}  
		
	logMessage(functionName +"publicKey: " + base64publicKey);
	logMessage(functionName + 'credentialId:' + base64credentialId);
	logMessage(functionName +"AAGUID: " + aaguid);
	
    // Prepare the data to send
    const data = {
		option: 'publicKey',
		aaguid: aaguid,
        userId: userHandle,
		challenge: challenge, // base64
        credentialId: base64credentialId, 
		clientDataJSON: base64clientDataJSON,
        publicKey: base64publicKey
		
    };

    // Send the data to the server using Fetch API
    return fetch(url, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        if (!response.ok) {
          throw new Error(functionName + 'Error 1! Status: ${response.status}'); }
	  
        return response.json();
    })
    .then(data => {
		if (data["status"] == "error")
		{ alert(functionName + 'Error: 2: ' + data["message"]);}
		else
		{ sessionStorage.webAuthn_authorised	= 'yes';
          logMessage(functionName + 'status: ' + data.status);			
          logMessage(functionName + 'message: ' + data.message);			
		}  
    })
	
    .catch(error => {
        logMessage(functionName + 'Error 3: ' + error);
     }
  );
}

function sendSignatureToServer(userHandle, challenge, signature, clientDataJSON, authenticatorData)
{
    const functionName = 'sendSignatureToServer: ';
    logMessage(functionName + 'Start');				

    const base64Signature = base64urlEncode(signature);
	const base64clientDataJSON = base64urlEncode(clientDataJSON);
    const base64authenticatorData = base64urlEncode(authenticatorData);

    logMessage("base64Signature:\n" + base64Signature);
    logMessage("base64clientDataJSON:\n" + base64clientDataJSON);
    logMessage("base64authenticatorData:\n" + base64authenticatorData);
		
    // Prepare the data to send
    const data = {
		option: 'signature',
		aaguid : localStorage.webauthn_AAGUID,
        userId: userHandle,
		challenge: challenge,
		signature: base64Signature,
		clientDataJSON: base64clientDataJSON,
		authenticatorData: base64authenticatorData
    };
	
// Send the data to the server using Fetch API
	try {
    return fetch(url, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        if (!response.ok) {
          throw new Error(functionName +'!: Status: ${response.status}'); }
	  
        return response.json();
    })
    .then(data => {
		console.log(functionName + 'Server response:', data);

		if (data["status"] == "error")
		{ throw new Error(functionName + "2: " + data["message"]);
		  }
		else
		  sessionStorage.webAuthn_authorised = 'yes';
    })
    .catch(error => {
        errMessage = "3: " + error.message;
		throw new Error(errMessage);
    });
	
  } catch (error) {
      errMessage = error.message;
	  logError(functionName + errMessage);
	  throw new Error(errMessage);
  }	
}




function logMessage(message)
{ console.log(message);}

function logError(message)
{ console.error(message);}


function base64ToUint8Array(base64) {
  const binaryString = window.atob(base64); // Decode Base64 to binary string
  const len = binaryString.length;
  const bytes = new Uint8Array(len);
  
  for (let i = 0; i < len; i++) {
    bytes[i] = binaryString.charCodeAt(i); // Convert each character to byte
  }
  
  return bytes;
}

function extractAuthData(attestationBuffer) {
  const bytes = new Uint8Array(attestationBuffer);

  for (let i = 0; i < bytes.length - 8; i++) {
    // Look for "authData" key: 0x68 + ASCII for 'a','u','t','h','D','a','t','a'
    if (
      bytes[i] === 0x68 &&
      bytes[i+1] === 0x61 && bytes[i+2] === 0x75 &&
      bytes[i+3] === 0x74 && bytes[i+4] === 0x68 &&
      bytes[i+5] === 0x44 && bytes[i+6] === 0x61 &&
      bytes[i+7] === 0x74 && bytes[i+8] === 0x61
    ) {
      const typeByte = bytes[i+9];
      let length = 0;
      let offset = i + 10;

      if (typeByte === 0x58) {
        length = bytes[offset];
        offset += 1;
      } else if (typeByte === 0x59) {
        length = (bytes[offset] << 8) + bytes[offset + 1];
        offset += 2;
      } else if (typeByte === 0x5A) {
        length = (bytes[offset] << 24) + (bytes[offset + 1] << 16) +
                 (bytes[offset + 2] << 8) + bytes[offset + 3];
        offset += 4;
      } else {
        console.error("Unsupported byte string format for authData");
        return null;
      }

      const authData = bytes.slice(offset, offset + length);
      return authData;
    }
  }

  return "AAGUID not known";
}

function extractAAGUID(attestationBuffer) {
  authData = extractAuthData(attestationBuffer);
  const aaguidBytes = authData.slice(37, 53);
  return [...aaguidBytes].map(b => b.toString(16).padStart(2, '0')).join('');
}

// Function to convert ArrayBuffer to Base64url
function base64urlEncode(data) {
    const binaryString = String.fromCharCode.apply(null, new Uint8Array(data));
    let base64 = btoa(binaryString);
    return base64.replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
}

function showPasskeyError(err)
{ 
  logMessage(err.name + " - " + err.message);
  
  if (err.name === "NotAllowedError") {
    emess = "User cancelled or timed out.";
  } else if (err.name === "InvalidStateError") {
    emess = "Credential ID not found on this device.";
  } else {
    extraMsg = isItImplemented(err);
	emess = err.name + ": " + err.message + extraMsg; 
  }
 
   alert (emess);
}  

function isItImplemented(msg)
{
  if (String(msg).toLowerCase().includes("not implemented"))
	return ('\n\nProblem could be caused by one of your Browser extensions');
  else
	return ('');  
}

function getBrowserInfo() {
    if (navigator.duckduckgo != undefined)
        return "DuckDuckGo Browser";
		
    const userAgent = navigator.userAgent;

    if (/SamsungBrowser/.test(userAgent)) {
        return "Samsung Internet Browser";
    } else if (/OnePlusBrowser/.test(userAgent)) {
        return "OnePlus Browser";
    } else if (/MiuiBrowser/.test(userAgent)) {
        return "Xiaomi Mi Browser";
    } else if (/NothingBrowser/.test(userAgent)) {
        return "Nothing Browser";
    } else if (/PixelBrowser/.test(userAgent)) {
        return "Google Pixel Browser";
    } else if (/HuaweiBrowser/.test(userAgent)) {
        return "Huawei Browser";
    } else if (/Chrome/.test(userAgent) && /Mobile/.test(userAgent)) {
		if (/Edg/.test(userAgent)) 
          return "Microsoft Edge on Mobile";
        else
	    if (/OPR/.test(userAgent) && /Touch/.test(userAgent)) 
          return "Opera Touch Browser";
 		else
	    if (/OPR/.test(userAgent)) 
          return "Opera Browser";
 		else
		return "Chrome on Mobile";
    } else if (/Chrome/.test(userAgent)) {
		if (/Edg/.test(userAgent)) 
          return "Microsoft Edge on Desktop";
        else
	    if (/OPR/.test(userAgent)) 
          return "Opera Browser";
 		else
          return "Chrome on Desktop";
	} else if (/Firefox/.test(userAgent) && /Mobile/.test(userAgent)) {
        return "Firefox on Mobile";
    } else if (/Firefox/.test(userAgent)) {
        return "Firefox on Desktop";
    } else if (/Safari/.test(userAgent) && /Mobile/.test(userAgent)) {
        return "Safari on Mobile";
    } else if (/Safari/.test(userAgent)) {
        return "Safari on Desktop";
    } else if (/Opera/.test(userAgent) || /OPR/.test(userAgent)) {
        return /Mobile/.test(userAgent) ? "Opera on Mobile" : "Opera on Desktop";
    } else if (/OPR/.test(userAgent) && /Touch/.test(userAgent)) {
        return "Opera Touch Browser";
    } else if (/MSIE/.test(userAgent) || /Trident/.test(userAgent)) {
        return "Internet Explorer";		
	} else {
        return userAgent;
    }
}

function centerElement() {
  const element = document.getElementById('webauthn_container');
  const viewportWidth = window.innerWidth;
  const elementWidth = element.offsetWidth;

  // Centering the element horizontally
  element.style.left = (viewportWidth / 2 - elementWidth / 2) + 'px'; 
}

